c Copyright (C) 2010 J.L. Llanio-Trujillo, J.M.C. Marques, F.B. Pereira
c Please cite: J.L. Llanio-Trujillo, J.M.C. Marques, F.B. Pereira,
c                         J. Phys. Chem. A (acepted).
c
c     ################################################################
c     ##                                                            ##
c     ##  function minrigid1  --  energy and gradient for minrigid  ##
c     ##                                                            ##
c     ################################################################
c
c
c     "minrigid1" is a service routine that computes the energy (minrigid1)
c     and gradient (g-vector) with respect to the search parameters (3 center
c     of mass and 3 Euler angles per rigid body).
c
c
      function minrigid1 (xx,qrb,nrb,nsite,g)  !input: xx,qrb  !output: minrigid1,g
      implicit real*8 (a-h,o-z)
      real*8 minrigid1
c      parameter (nsite=4) 
      dimension qrb(3*nsite,nrb)
      dimension rbc(6,nrb)
      dimension q(3*nsite,nrb)
      dimension xx(6*nrb)
      dimension g(6*nrb)
      dimension derivs_rbc(6,nrb)   

c
c     translate optimization parameters (xx-vector) to rigid-body coordinates (rbc)
c
      nvar = 0
      do i = 1, nrb
         do j = 1, 6
            nvar = nvar + 1
            rbc(j,i) = xx(nvar)
         end do
      end do
c
c     compute and store the energy and gradient from the body-frame referenced
c     Cartesian coordinates (qrb) and the current solution vector (rbc). For
c     that, one needs to transform from the body-frame referenced coordinates (qrb)
c     to the Laboratory-frame referenced ones (q), in which the potential and
c     derivatives are expressed.
c
      call rigidxyz(qrb,rbc,nrb,nsite,q)  !input: qrb,rbc   !output: q
      call gradrgd (q,qrb,rbc,nrb,nsite,energy,derivs_rbc)   !input: q,qrb,rbc   !output: energy,derivs_rbc
      minrigid1 = energy
c
c     store rigid body gradient as optimization gradient
c
      nvar = 0
      do i = 1, nrb
         do j = 1, 6
            nvar = nvar + 1
            g(nvar) = derivs_rbc(j,i)
         end do
      end do
      return
      end


c
c
c     ###############################################################
c     ##                                                           ##
c     ##  subroutine rigidxyz  --  rigid body to Cartesian coords  ##
c     ##                                                           ##
c     ###############################################################
c
c
c     "rigidxyz" computes Cartesian coordinates for a rigid body
c     group via rotation and translation of reference coordinates
c
c     literature reference:
c
c     Herbert Goldstein, "Classical Mechanics, 2nd Edition",
c     Addison-Wesley, Reading, MA, 1980; see the Euler angle
c     xyz convention in Appendix B
c
c
      subroutine rigidxyz(qrb,rbc,nrb,nsite,q)  !input: qrb,rbc   !output: q
      implicit real*8 (a-h,o-z)
c      parameter (nsite=4) 
      dimension q(3*nsite,nrb)
      dimension qrb(3*nsite,nrb),rbc(6,nrb) 
      dimension a(3,3)
c
c
c     get the center of mass and Euler angles for each group
c
      do i = 1, nrb
         xcm = rbc(1,i)
         ycm = rbc(2,i)
         zcm = rbc(3,i)
         phi = rbc(4,i)
         theta = rbc(5,i)
         psi = rbc(6,i)
         cphi = cos(phi)
         sphi = sin(phi)
         ctheta = cos(theta)
         stheta = sin(theta)
         cpsi = cos(psi)
         spsi = sin(psi)
c
c     construct the rotation matrix from Euler angle values
c
         a(1,1) = ctheta * cphi
         a(2,1) = spsi*stheta*cphi - cpsi*sphi
         a(3,1) = cpsi*stheta*cphi + spsi*sphi
         a(1,2) = ctheta * sphi
         a(2,2) = spsi*stheta*sphi + cpsi*cphi
         a(3,2) = cpsi*stheta*sphi - spsi*cphi
         a(1,3) = -stheta
         a(2,3) = ctheta * spsi
         a(3,3) = ctheta * cpsi
c
c     rotate and translate reference coordinates into global frame
c
            do k=1,nsite
               k3=3*k
               k2=k3-1
               k1=k2-1
            xterm = qrb(k1,i)
            yterm = qrb(k2,i)
            zterm = qrb(k3,i)
            q(k1,i) = a(1,1)*xterm + a(2,1)*yterm + a(3,1)*zterm + xcm
            q(k2,i) = a(1,2)*xterm + a(2,2)*yterm + a(3,2)*zterm + ycm
            q(k3,i) = a(1,3)*xterm + a(2,3)*yterm + a(3,3)*zterm + zcm
         end do
      end do
      return
      end


c ********************************************************************************************************
      subroutine gradrgd (q,qrb,rbc,nrb,nsite,energy,derivs_rbc)  !input: q,qrb,rbc   !output: energy,derivs_rbc
c ********************************************************************************************************
      implicit real*8 (a-h,o-z)
c      parameter (nsite=4)
      dimension q(3*nsite,nrb),qrb(3*nsite,nrb),rbc(6,nrb)
      dimension derivs_xyz(3*nsite,nrb)
      dimension derivs_rbc(6,nrb)


c   zero out the rigid-body derivatives (derivs_rbc)

      do i = 1, nrb
         do j = 1, 6
            derivs_rbc(j,i) = 0.0d0
         end do
      end do

c  calculate the energy (energy) and Cartesian first derivatives (derivs_xyz)

      call gradient (q,nrb,nsite,energy,derivs_xyz)  !input: q   !output: energy,derivs_xyz

c   copy the rbc's (6 in total: 3 translation params + 3 orientation params)
c   for each rigid body (loop running index --> i)
      do i = 1, nrb       !outer loop
         xcm   = rbc(1,i)
         ycm   = rbc(2,i)
         zcm   = rbc(3,i)
         phi   = rbc(4,i)
         theta = rbc(5,i)
         psi   = rbc(6,i)

         cphi = cos(phi)
         sphi = sin(phi)
         ctheta = cos(theta)
         stheta = sin(theta)
         cpsi   = cos(psi)
         spsi   = sin(psi)


c R rotation matrix (XYZ convention)

c  DR/Dphi matrix
         dphi_rxx = -ctheta * sphi
         dphi_ryx = -spsi*stheta*sphi - cpsi*cphi
         dphi_rzx = -cpsi*stheta*sphi + spsi*cphi
         dphi_rxy = ctheta * cphi
         dphi_ryy = spsi*stheta*cphi - cpsi*sphi
         dphi_rzy = cpsi*stheta*cphi + spsi*sphi
         dphi_rxz = 0.0d0
         dphi_ryz = 0.0d0
         dphi_rzz = 0.0d0
c  DR/Dtheta matrix        
         dtheta_rxx = -stheta * cphi
         dtheta_ryx = spsi*ctheta*cphi
         dtheta_rzx = cpsi*ctheta*cphi
         dtheta_rxy = -stheta * sphi
         dtheta_ryy = spsi*ctheta*sphi
         dtheta_rzy = cpsi*ctheta*sphi
         dtheta_rxz = -ctheta
         dtheta_ryz = -stheta * spsi
         dtheta_rzz = -stheta * cpsi
c  DR/Dpsi matrix
         dpsi_rxx = 0.0d0
         dpsi_ryx = cpsi*stheta*cphi + spsi*sphi
         dpsi_rzx = -spsi*stheta*cphi + cpsi*sphi
         dpsi_rxy = 0.0d0
         dpsi_ryy = cpsi*stheta*sphi - spsi*cphi
         dpsi_rzy = -spsi*stheta*sphi - cpsi*cphi
         dpsi_rxz = 0.0d0
         dpsi_ryz = ctheta * cpsi
         dpsi_rzz = -ctheta * spsi


c  First:
c  rigid-body gradients of the Interaction for translations:
c derivs_rbc(1,i), derivs_rbc(2,i) & derivs_rbc(3,i)

         do k = 1, nsite
                k3=3*k
                k2=k3-1
                k1=k2-1
            derivs_rbc(1,i) = derivs_rbc(1,i) + derivs_xyz(k1,i)
            derivs_rbc(2,i) = derivs_rbc(2,i) + derivs_xyz(k2,i)
            derivs_rbc(3,i) = derivs_rbc(3,i) + derivs_xyz(k3,i)
         end do


c   rigid-body gradient w.r.t first 
c   (rotation about Z-axis an angle with magnitude --> phi)
c   orientation parameter derivs_rbc(4,i)

        do k=1,nsite
           k3=3*k
           k2=k3-1
           k1=k2-1
           x0=qrb(k1,i)
           y0=qrb(k2,i)
           z0=qrb(k3,i)

           dxdphi=dphi_rxx*x0+dphi_ryx*y0+dphi_rzx*z0
           dydphi=dphi_rxy*x0+dphi_ryy*y0+dphi_rzy*z0
           dzdphi=dphi_rxz*x0+dphi_ryz*y0+dphi_rzz*z0

           derivs_rbc(4,i)=derivs_rbc(4,i)+derivs_xyz(k1,i)*dxdphi+
     >              derivs_xyz(k2,i)*dydphi+derivs_xyz(k3,i)*dzdphi
        enddo


c   rigid-body gradient w.r.t second 
c   (rotation about an intermediary Y-axis an angle with magnitude --> theta)
c   orientation parameter derivs_rbc(5,i)

        do k=1,nsite
           k3=3*k
           k2=k3-1
           k1=k2-1
           x0=qrb(k1,i)
           y0=qrb(k2,i)
           z0=qrb(k3,i)

           dxdtheta=dtheta_rxx*x0+dtheta_ryx*y0+dtheta_rzx*z0
           dydtheta=dtheta_rxy*x0+dtheta_ryy*y0+dtheta_rzy*z0
           dzdtheta=dtheta_rxz*x0+dtheta_ryz*y0+dtheta_rzz*z0

           derivs_rbc(5,i)=derivs_rbc(5,i)+derivs_xyz(k1,i)*dxdtheta+
     >            derivs_xyz(k2,i)*dydtheta+derivs_xyz(k3,i)*dzdtheta
        enddo

c  Finally:
c  rigid-body gradient w.r.t third (rotation about X-axis an angle psi)
c  orientation parameter derivs_rbc(6,i)


        do k=1,nsite
           k3=3*k
           k2=k3-1
           k1=k2-1
           x0=qrb(k1,i)
           y0=qrb(k2,i)
           z0=qrb(k3,i)

           dxdpsi=dpsi_rxx*x0+dpsi_ryx*y0+dpsi_rzx*z0
           dydpsi=dpsi_rxy*x0+dpsi_ryy*y0+dpsi_rzy*z0
           dzdpsi=dpsi_rxz*x0+dpsi_ryz*y0+dpsi_rzz*z0

           derivs_rbc(6,i)=derivs_rbc(6,i)+derivs_xyz(k1,i)*dxdpsi+
     >              derivs_xyz(k2,i)*dydpsi+derivs_xyz(k3,i)*dzdpsi
        enddo
      enddo
      return
      end

c *****************************************************************
c Total energy & Cartesian derivatives (TIP4P water potential)
c functional form & parameters from:
c Jorgensen WL et.al; J.Chem Phys. 79(2)1983 (see TABLE I)
c *****************************************************************
      subroutine gradient (q,nrb,nsite,energy,derivs)  !user defined potential
      implicit real*8 (a-h,o-z)
c      parameter (nsite=4)  
      dimension q(3*nsite,nrb)
      dimension derivs(3*nsite,nrb)
      dimension deww_cc(3*nsite,nrb)
      dimension deww_lj(3*nsite,nrb)


c zero out potential energy components
      eww_cc=0.0d0
      eww_lj=0.0d0
      
c zero out the drivatives

      do Irb = 1, nrb
           do i = 1, 3*nsite
              deww_cc(i,Irb) = 0.0d0
              deww_lj(i,Irb) = 0.0d0
           end do
      end do

c Coulomb energy & derivatives
      call v_w_w_colmb (q,nrb,nsite,eww_cc,deww_cc)

c LJ energy & derivatives
      call v_w_w_lj (q,nrb,nsite,eww_lj,deww_lj)


      energy= eww_cc + eww_lj


c sum-up the total first cartesian derivatives

      do Irb = 1, nrb
           do i = 1, 3*nsite
              dertot=deww_cc(i,Irb)+deww_lj(i,Irb)
              derivs(i,Irb)=dertot
           end do
      end do
      
      return
      end

c *******************************************************
c    TIP4P potential (Coulomb component)
c parameters from:
c !Jorgensen WL et.al; J.Chem Phys. 79(2)1983 (see TABLE I)
c *******************************************************
      subroutine v_w_w_colmb (q,nrb,nsite,v,derivs)
      implicit real*8 (a-h,o-z)
c      parameter (nsite=4) 
      dimension derivs(3*nsite,nrb)
      dimension q(3*nsite,nrb)
      dimension pchg(4)
      DATA (pchg(m),m=1,4)/0.0d0,
     >                     0.52d0,0.52d0,
     >                    -1.04d0/


c zero out the potential         
        v=0.0d0

c 4-sites water
c        
        do Jw=1,nrb-1
           do Kw=Jw+1,nrb
              do j=1,nsite  
                 j3=3*j
                 j2=j3-1
                 j1=j2-1 

                 do k=1,nsite
                     k3=3*k
                     k2=k3-1
                     k1=k2-1

                     t1= q(k1,Kw)-q(j1,Jw)
                     t2= q(k2,Kw)-q(j2,Jw)
                     t3= q(k3,Kw)-q(j3,Jw)
                     rjk= dsqrt(t1*t1+t2*t2+t3*t3)
                      
                     vjk=(1389.354848D0/4.184d0)*pchg(j)*pchg(k)/rjk  !in kcal/mol 
                     v=v+vjk
                 enddo
              enddo
           enddo
        enddo

c zero out the derivatives        
        do Irb = 1, nrb
           do j = 1, 3*nsite
              derivs(j,Irb) = 0.0d0
           end do
        end do


        do Iw=1,nrb-1
         do Jw=Iw+1,nrb

c derivatives w.r.t. i-sites (on Iw) coordinates 
               do i=1,nsite
                  i3=3*i
                  i2=i3-1
                  i1=i2-1 
                 do j=1,nsite
                     j3=3*j
                     j2=j3-1
                     j1=j2-1 

                     t1= q(j1,Jw)-q(i1,Iw)
                     t2= q(j2,Jw)-q(i2,Iw)
                     t3= q(j3,Jw)-q(i3,Iw)
                     rij2= t1*t1+t2*t2+t3*t3
                     rij= dsqrt(rij2)
                     rij3=rij2*rij

          dum1=(1389.354848D0/4.184d0)*pchg(i)*pchg(j)
          dum2=-1.0d0/rij3
          dum=dum1*dum2 

         derivs(i1,Iw)=derivs(i1,Iw)-t1*dum
         derivs(i2,Iw)=derivs(i2,Iw)-t2*dum
         derivs(i3,Iw)=derivs(i3,Iw)-t3*dum


                 enddo
               enddo 
c derivatives w.r.t. j-sites (on Jw) coordinates 
            do j=1,nsite
               j3=3*j
               j2=j3-1
               j1=j2-1 
                do i=1,nsite
                    i3=3*i
                    i2=i3-1
                    i1=i2-1 

               t1= q(j1,Jw)-q(i1,Iw)
               t2= q(j2,Jw)-q(i2,Iw)
               t3= q(j3,Jw)-q(i3,Iw)
              rij2= t1*t1+t2*t2+t3*t3
              rij= dsqrt(rij2)
              rij3=rij2*rij
         
          dum1=(1389.354848D0/4.184d0)*pchg(i)*pchg(j)
          dum2=-1.0d0/rij3
          dum=dum1*dum2 


         derivs(j1,Jw)=derivs(j1,Jw)+t1*dum
         derivs(j2,Jw)=derivs(j2,Jw)+t2*dum
         derivs(j3,Jw)=derivs(j3,Jw)+t3*dum

                enddo
            enddo 
         enddo 
        enddo 

        return
        end


c *********************************************************
c    TIP4P potential (LJ component)
c parameters from:
c !Jorgensen WL et.al; J.Chem Phys. 79(2)1983 (see TABLE I)
c *********************************************************
      subroutine v_w_w_lj (q,nrb,nsite,v,derivs)
      implicit real*8 (a-h,o-z)
c      parameter (nsite=4)
      dimension derivs(3*nsite,nrb)
      dimension q(3*nsite,nrb)
      Data A,C/6.0d5,6.1d2/   


c zero out the potential         
        v=0.0d0

c 4-sites water
c
        do Jw=1,nrb-1
           do Kw=Jw+1,nrb
c              do j=1,nsite  !only sites 1 (Oxygen) on each Jw,Kw monomer, interact via Lennard-Jones
               j=1
                 j3=3*j
                 j2=j3-1
                 j1=j2-1 

c                 do k=1,nsite
                  k=1
                     k3=3*k
                     k2=k3-1
                     k1=k2-1

                     t1= q(k1,Kw)-q(j1,Jw)
                     t2= q(k2,Kw)-q(j2,Jw)
                     t3= q(k3,Kw)-q(j3,Jw)

                     rjk2= t1*t1+t2*t2+t3*t3

                     p6=1.0d0/rjk2**3
                     p12=p6*p6

                     vjk=p6*(A*p6-C)
                     v=v+vjk
c                 enddo
c              enddo
           enddo
        enddo

c zero out the derivatives        
        do Irb = 1, nrb
           do j = 1, 3*nsite
              derivs(j,Irb) = 0.0d0
           end do
        end do


        do Iw=1,nrb-1
         do Jw=Iw+1,nrb

c derivatives w.r.t. i-sites (on Iw) coordinates
c            do i=1,nsite
             i=1
               i3=3*i
               i2=i3-1
               i1=i2-1 
c               do j=1,nsite
                j=1 
                
                   j3=3*j
                   j2=j3-1
                   j1=j2-1 

                   t1= q(j1,Jw)-q(i1,Iw)
                   t2= q(j2,Jw)-q(i2,Iw)
                   t3= q(j3,Jw)-q(i3,Iw)
                   rij= dsqrt(t1*t1+t2*t2+t3*t3)
         
                  dum1=12.0d0*A/rij**14
                  dum2=-6.0d0*C/rij**8

       derivs(i1,Iw)=derivs(i1,Iw)+t1*(dum1+dum2)
       derivs(i2,Iw)=derivs(i2,Iw)+t2*(dum1+dum2)
       derivs(i3,Iw)=derivs(i3,Iw)+t3*(dum1+dum2)


c               enddo
c            enddo 
c derivatives w.r.t. j-sites (on Jw) coordinates 
c            do j=1,nsite
             j=1
               j3=3*j
               j2=j3-1
               j1=j2-1 
c               do i=1,nsite
                i=1
                  i3=3*i
                  i2=i3-1
                  i1=i2-1 

                  t1= q(j1,Jw)-q(i1,Iw)
                  t2= q(j2,Jw)-q(i2,Iw)
                  t3= q(j3,Jw)-q(i3,Iw)
                  rij= dsqrt(t1*t1+t2*t2+t3*t3)
         
                  dum1=12.0d0*A/rij**14
                  dum2=-6.0d0*C/rij**8

      derivs(j1,Jw)=derivs(j1,Jw)-t1*(dum1+dum2)
      derivs(j2,Jw)=derivs(j2,Jw)-t2*(dum1+dum2)
      derivs(j3,Jw)=derivs(j3,Jw)-t3*(dum1+dum2)

c               enddo
c            enddo 
         enddo 
        enddo 

        return
        end
