#include <math.h>
#include <stdlib.h>
#include <time.h>
#include "crossover.h"
#include "evaluation.h"
#include "mutation.h"
#include "utils.h"


/* Crossover Types

 	0: Single point crossover
 	1: Double point Crossover
	2: BLX-alpha crossover
 	3: SBX (Simulated Binary Crossover - proposed by Deb)
*/


void single_point_crossover(int ind_size, double p1[], double p2[], double off1[], double off2[])
{
	int i, position;

	position = random_number(1, ind_size-2);

	for(i = 0; i < position; i++){

		off1[i] = p1[i];
		off2[i] = p2[i];
	}

	for(i = position; i < ind_size; i++){

		off1[i] = p2[i];
		off2[i] = p1[i];
	}
}


void double_point_crossover(int ind_size, double p1[], double p2[], double off1[], double off2[])
{
	int i, position1, position2, temp;

	position1 = random_number(1, ind_size-2);
	temp = random_number(1, ind_size-2);

	if(position1 < temp){

		position2 = temp;
	}else{

		position2 = position1;
		position1 = temp;
	}

	for(i = 0; i < position1; i++){

		off1[i] = p1[i];
		off2[i] = p2[i];
	}

	for(i = position1; i < position2; i++){

		off1[i] = p2[i];
		off2[i] = p1[i];
	}

	for(i = position2; i < ind_size; i++){

			off1[i] = p1[i];
			off2[i] = p2[i];
	}
}


#define ETA_SBX	3.0

void SBX_crossover(int ind_size, double p1[], double p2[], double off1[], double off2[]){

	int i;
	double mu, beta;
	
	do{
		mu = uniform01();
	}while( mu > 0.99);
	
	if(mu <= 0.5)
	{
		beta = 2*mu*(1.0/(ETA_SBX+1.0));
	}
	else
	{
		beta = pow( 1.0/(2.0*(1-mu)), (1.0/(ETA_SBX+1.0)));
	}
	
	for(i = 0; i < ind_size; i++)
	{
		off1[i] = 0.5*((1.0+beta)*p1[i] + (1.0-beta)*p2[i]);
		off2[i] = 0.5*((1.0-beta)*p1[i] + (1.0+beta)*p2[i]);
	}
}



void BLX_alpha_crossover(int ind_size, double p1[], double p2[], double off1[], double off2[], double blx_alpha){


	double max, min, dif;
	int i;

	for(i = 0; i < ind_size; i++){

		max = p1[i];
		min = p2[i];

		if(p2[i] > max){
			max = p2[i];
			min = p1[i];
		}

		dif = max - min;

		min = min - (dif*blx_alpha);
		max = max + (dif*blx_alpha);

		off1[i] = min + (uniform01()*(max - min));
		off2[i] = min + (uniform01()*(max - min));
		

	}
}



// applies crossover in sequential ordering. generational GA.
void apply_crossover(p_ind population, int pop_size, int ind_size, double p_cx, int type,
	double blx_alpha, vect_domain bounds, int dir) {

	int i;	           		        // positions
	ind c1, c2, c3;

	// iterates population and produce offsprings
	for (i = 0; i < pop_size; i += 2)
	{
		if(flip(p_cx) == TRUE)
		{

			c1.genes = malloc(sizeof(double) * ind_size);
			c2.genes = malloc(sizeof(double) * ind_size);

				switch(type){
					case 0:
						  single_point_crossover(ind_size, population[i].genes, population[i+1].genes, c1.genes, c2.genes);
						  break;
					case 1:
						  double_point_crossover(ind_size, population[i].genes, population[i+1].genes, c1.genes, c2.genes);
						  break;
					case 2:
						  BLX_alpha_crossover(ind_size, population[i].genes, population[i+1].genes, c1.genes, c2.genes, blx_alpha);
						  break;
					case 3:
						  SBX_crossover(ind_size, population[i].genes, population[i+1].genes, c1.genes, c2.genes);
						  break;
					default: puts("Crossover type error"); break;
				}

				free(population[i].genes);
				free(population[i+1].genes);

				population[i] = c1;
				population[i+1] = c2;

		}
	}
	

	bounds_control(population, pop_size, ind_size, bounds);
}





